<?php
// leak_investigation.php
session_start();
require 'db_connection.php'; // Your PDO connection

// 1. Strict Security: Only Admins or high-level Managers can access this
if (!isset($_SESSION['user_id']) || $_SESSION['role_id'] > 2) { 
    // Assuming role_id 1 is Admin, 2 is Manager
    die("Unauthorized access. This incident has been logged.");
}

$investigation_results = null;
$error_message = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['leaked_file'])) {
    
    $uploaded_file = $_FILES['leaked_file'];
    
    if ($uploaded_file['error'] === UPLOAD_ERR_OK) {
        
        $temp_file = $uploaded_file['tmp_name'];
        $zip = new ZipArchive;
        
        // 2. Open the uploaded Word/PPT file as a ZIP archive
        if ($zip->open($temp_file) === TRUE) {
            
            // 3. Hunt for the hidden NCDC payload
            // We check both Word and PPT default media paths
            $xml_content = false;
            if ($zip->locateName('word/media/ncdc_sys_data.xml') !== false) {
                $xml_content = $zip->getFromName('word/media/ncdc_sys_data.xml');
            } elseif ($zip->locateName('ppt/media/ncdc_sys_data.xml') !== false) {
                $xml_content = $zip->getFromName('ppt/media/ncdc_sys_data.xml');
            }
            
            $zip->close();
            
            if ($xml_content) {
                // 4. Parse the extracted XML
                $tracker_data = simplexml_load_string($xml_content);
                
                if ($tracker_data) {
                    // Extract variables safely
                    $share_id = (int)$tracker_data->share_id;
                    $recipient = htmlspecialchars((string)$tracker_data->recipient);
                    $email = htmlspecialchars((string)$tracker_data->email);
                    $downloaded_at = htmlspecialchars((string)$tracker_data->downloaded_at);
                    $ip_address = htmlspecialchars((string)$tracker_data->ip_address);
                    $auth_token = (string)$tracker_data->ncdc_auth_token;
                    
                    // 5. Cross-reference with the secure database to verify authenticity
                    // This prevents someone from framing another user by faking the XML
                    $verify_stmt = $pdo->prepare("SELECT document_id FROM document_shares WHERE share_id = ? AND recipient_email = ?");
                    $verify_stmt->execute([$share_id, $email]);
                    $db_record = $verify_stmt->fetch();
                    
                    if ($db_record) {
                        $expected_token = hash('sha256', $share_id . $email . 'SECRET_SALT');
                        
                        if (hash_equals($expected_token, $auth_token)) {
                            // MATCH FOUND! 
                            $investigation_results = [
                                'status' => 'VERIFIED MATCH',
                                'recipient' => $recipient,
                                'email' => $email,
                                'downloaded_at' => $downloaded_at,
                                'ip_address' => $ip_address,
                                'document_id' => $db_record['document_id']
                            ];
                        } else {
                            $error_message = "Payload found, but cryptographic signature failed. The tracker may have been tampered with.";
                        }
                    } else {
                        $error_message = "Payload found, but no matching record exists in the NCDC database.";
                    }
                } else {
                    $error_message = "Hidden file found, but data is corrupted.";
                }
            } else {
                $error_message = "CLEAN FILE: No NCDC tracking payload found in this document.";
            }
        } else {
            $error_message = "Error: Could not process the file. Ensure it is a valid .docx or .pptx format.";
        }
    } else {
        $error_message = "Upload error. Please try again.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <title>NCDC | Forensic Leak Investigation</title>
    <style>
        body { font-family: 'Inter', sans-serif; background-color: #F4F7F9; padding: 40px; color: #1E293B; }
        .dashboard { max-width: 800px; margin: 0 auto; background: white; padding: 30px; border-radius: 8px; box-shadow: 0 4px 6px rgba(0,0,0,0.05); border-top: 4px solid #CE1126; }
        h1 { color: #CE1126; margin-bottom: 5px; }
        p.subtitle { color: #64748B; margin-bottom: 30px; }
        .upload-zone { border: 2px dashed #CBD5E1; padding: 40px; text-align: center; border-radius: 8px; margin-bottom: 30px; background: #F8FAFC; }
        input[type="file"] { margin-top: 15px; }
        .btn-danger { background: #CE1126; color: white; padding: 12px 24px; border: none; border-radius: 6px; font-weight: bold; cursor: pointer; margin-top: 20px; }
        .btn-danger:hover { background: #A90E1F; }
        
        .results-box { background: #FEF2F2; border: 1px solid #FCA5A5; padding: 20px; border-radius: 8px; margin-top: 20px; }
        .results-box h3 { color: #991B1B; margin-top: 0; }
        .data-grid { display: grid; grid-template-columns: 1fr 2fr; gap: 10px; margin-top: 15px; }
        .data-label { font-weight: bold; color: #7F1D1D; }
        .data-value { font-family: monospace; font-size: 14px; background: white; padding: 4px 8px; border-radius: 4px; border: 1px solid #FECACA; }
        
        .alert-box { background: #F1F5F9; border: 1px solid #CBD5E1; padding: 15px; border-radius: 6px; margin-top: 20px; color: #475569; }
    </style>
</head>
<body>

<div class="dashboard">
    <h1>Forensic Document Analysis</h1>
    <p class="subtitle">Upload a suspected leaked file (.docx or .pptx) to extract the hidden NCDC security signature.</p>

    <form method="POST" enctype="multipart/form-data">
        <div class="upload-zone">
            <h3>Select Leaked File for Scanning</h3>
            <input type="file" name="leaked_file" accept=".docx, .pptx" required>
            <br>
            <button type="submit" class="btn-danger">Run Forensic Extraction</button>
        </div>
    </form>

    <?php if (isset($investigation_results)): ?>
        <div class="results-box">
            <h3><i class="fas fa-exclamation-triangle"></i> CRITICAL: LEAK SOURCE IDENTIFIED</h3>
            <div class="data-grid">
                <div class="data-label">Originating Target:</div>
                <div class="data-value"><?php echo $investigation_results['recipient']; ?></div>
                
                <div class="data-label">Registered Email:</div>
                <div class="data-value"><?php echo $investigation_results['email']; ?></div>
                
                <div class="data-label">Time of Download:</div>
                <div class="data-value"><?php echo $investigation_results['downloaded_at']; ?> (Server Time)</div>
                
                <div class="data-label">Network IP Address:</div>
                <div class="data-value"><?php echo $investigation_results['ip_address']; ?></div>
                
                <div class="data-label">Internal DB Status:</div>
                <div class="data-value"><?php echo $investigation_results['status']; ?></div>
            </div>
            <p style="margin-top: 15px; font-size: 12px; color: #991B1B;">* This signature has been cryptographically verified against the active database.</p>
        </div>
    <?php endif; ?>

    <?php if (isset($error_message)): ?>
        <div class="alert-box">
            <strong>Analysis Result:</strong> <?php echo htmlspecialchars($error_message); ?>
        </div>
    <?php endif; ?>

</div>

</body>
</html>